/**
 * \file: ipu_dp_csc.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * Compute and update IPU display processor color space (CSC) matrix w.r.t
 * change in color property like Hue, Saturation, Brightness, Contrast and
 * respective offset.
 *
 * \component: IPU
 *
 * \author: Gautham Kantharaju / RBEI / ECF3 / Gautham.Kantharaju@in.bosch.com
 *
 * \copyright: (c) 2015 ADIT Corporation
 *
 * \history
 * 0.1 Gautham Kantharaju Initial version
 ******************************************************************************/

/* PRQA: Lint Message 524: deactivation because only integer CSC matrix value needs to passed to IPU-v3 driver */
/*lint -save -e524 */

#include "ipu_dp_csc.h"

#if 0
#define DEBUG_OTHERS
#define DEBUG_CSC_A_MATRIX
#define DEBUG_HUE_MATRIX
#define DEBUG_CSC_B_MATRIX
#define DEBUG_CSC_A_B_MATRIX
#define DEBUG_CLR_PROPERTY
#endif

#define SUCCESS		0
#define ERROR		-1

#define CSC_MAX_VALUE				256
#define CSC_MIN_VALUE				0
#define CSC_EXPONENT_VAL			2
#define CSC_B_MATRIX_MAX_VALUE		1020
#define DIVIDE_BY_HUNDRED			100
#define SIGNED_FOURTEEN_BIT_VAL		0x3FFF

/* Color property max and min values */
#define SATURATION_MAX_VAL			100

#define HUE_MAX_VAL					180
#define HUE_MIN_VAL					-180

#define BRIGHTNESS_MAX_VAL			100
#define BRIGHTNESS_MIN_VAL			0

#define CONTRAST_MAX_VAL			100
#define CONTRAST_MIN_VAL			0

/* Offset values */
#define HUE_OFF_MAX_VAL				180
#define SATURATION_OFF_MAX_VAL		100
#define BRIGHTNESS_OFF_MAX_VAL		100

#define HUE_OFF_MIN_VAL				-180
#define SATURATION_OFF_MIN_VAL		0
#define BRIGHTNESS_OFF_MIN_VAL		-100

#define HUE_RANGE_VAL				60

#define MAX_CRTC_ID					32

/* Default values */
#define DEFAULT_HUE_VAL				0
#define DEFAULT_SATURATION_VAL		100
#define DEFAULT_BRIGHTNESS_VAL		100
#define DEFAULT_CONTRAST_VAL		100

#define DEFAULT_HUE_OFF_VAL			0
#define DEFAULT_SATURATION_OFF_VAL	0
#define DEFAULT_BRIGHTNESS_OFF_VAL	0

typedef struct _priv
{
	bool hue_off_update;
	bool csc_a_matrix_update;
	bool csc_b_matrix_update;
	bool contrast_update;
	float saturation;
	float brightness;
	float contrast;
	float cos_val;
	float sin_val;
	CLRSPACETYPE type;
	CLRPROP cprop;
	uint32_t crtc_count;
	uint32_t crtc_id[MAX_CRTC_ID];
	bool crtc_update;
}PRIVATE;

/* Global space */
PRIVATE g_priv;

void initialize_clrprop_priv (CLRPROP *out, PRIVATE *priv)
{
	int i = 0;
	int j = 0;

	if ((priv != NULL) && (out != NULL))
	{
		priv->hue_off_update = false;
		priv->csc_a_matrix_update = false;
		priv->csc_b_matrix_update = false;
		priv->contrast_update = false;
		priv->cos_val = 0;
		priv->sin_val = 0;

		if (INVALID_VALUE == priv->cprop.csc.crtc_id)
		{
			priv->crtc_count = 0;
			priv->crtc_update = true;

			/*
			 * Initially set crtc id to invalid value i.e 0xFF
			 * later valid crtc id will be retrieved and updated
			 */
			for (i = 0; i < MAX_CRTC_ID; i++)
				priv->crtc_id[i] = INVALID_VALUE;

			/*
			 * Initially set supported crtc id to invalid value
			 * i.e 0xFF, later supported crtc id will be updated
			 */
			for (i = 0; i < MAX_SUPPORTED_CRTC_ID; i++)
				out->supported_crtc_id[i] = INVALID_VALUE;
		}
		else
			priv->crtc_update = false;

		/*
		 * Everytime before matrix computation
		 * need to reset CSC 'A' and 'B' matrix value
		 */
		for(i = 0; i < 3; i++)
		{
			for(j = 0; j < 3; j++)
				out->csc.A[i][j] = 0;

			out->csc.B[i] = 0;
		}
	}
}

void correct_color_property_val(CLRPROP *out, PRIVATE *priv)
{
	if (INVALID_VALUE != out->hue)
	{
		if (out->hue > HUE_MAX_VAL)
			out->hue = HUE_MAX_VAL;

		if (out->hue < HUE_MIN_VAL)
			out->hue = HUE_MIN_VAL;
	}
	else
		/* Restore the previous value */
		out->hue = priv->cprop.hue;

	if (INVALID_VALUE != out->saturation)
	{
		if (out->saturation < CSC_MIN_VALUE)
			out->saturation = CSC_MIN_VALUE;

		if (out->saturation > SATURATION_MAX_VAL)
			out->saturation = SATURATION_MAX_VAL;
	}
	else
		/* Restore the previous value */
		out->saturation = priv->cprop.saturation;

	if (INVALID_VALUE != out->brightness)
	{
		if (out->brightness < BRIGHTNESS_MIN_VAL)
			out->brightness = BRIGHTNESS_MIN_VAL;

		if (out->brightness > BRIGHTNESS_MAX_VAL)
			out->brightness = BRIGHTNESS_MAX_VAL;
	}
	else
		/* Restore the previous value */
		out->brightness = priv->cprop.brightness;

	if (INVALID_VALUE != out->contrast)
	{
		if (out->contrast < CONTRAST_MIN_VAL)
			out->contrast = CONTRAST_MIN_VAL;

		if (out->contrast > CONTRAST_MAX_VAL)
			out->contrast = CONTRAST_MAX_VAL;
	}
	else
		/* Restore the previous value */
		out->contrast = priv->cprop.contrast;

	if (INVALID_VALUE != out->hue_off)
	{
		if (out->hue_off > HUE_OFF_MAX_VAL)
			out->hue_off = HUE_OFF_MAX_VAL;

		if (out->hue_off < HUE_OFF_MIN_VAL)
			out->hue_off = HUE_OFF_MIN_VAL;
	}
	else
		/* Restore the previous value */
		out->hue_off = priv->cprop.hue_off;

	if (INVALID_VALUE != out->saturation_off)
	{
		if (out->saturation_off > SATURATION_OFF_MAX_VAL)
			out->saturation_off = SATURATION_OFF_MAX_VAL;

		if (out->saturation_off < SATURATION_OFF_MIN_VAL)
			out->saturation_off = SATURATION_OFF_MIN_VAL;
	}
	else
		/* Restore the previous value */
		out->saturation_off = priv->cprop.saturation_off;

	if (INVALID_VALUE != out->brightness_off)
	{
		if (out->brightness_off > BRIGHTNESS_OFF_MAX_VAL)
			out->brightness_off = BRIGHTNESS_OFF_MAX_VAL;

		if (out->brightness_off < BRIGHTNESS_OFF_MIN_VAL)
			out->brightness_off = BRIGHTNESS_OFF_MIN_VAL;
	}
	else
		/* Restore the previous value */
		out->brightness_off = priv->cprop.brightness_off;

	if (!priv->hue_off_update)
		if (out->saturation_off > SATURATION_OFF_MIN_VAL)
			/*
			 * Compute CSC 'B' matrix w.r.t change in hue offset value
			 * only when saturation offset is greater then it's default
			 * value, otherwise change in hue offset value will have no
			 * effect.
			 */
			priv->hue_off_update = true;
}

void div_color_prop (CLRPROP *out, PRIVATE *priv)
{
	priv->saturation = ((float)out->saturation / DIVIDE_BY_HUNDRED);
	priv->brightness = ((float)out->brightness / DIVIDE_BY_HUNDRED);
	priv->contrast = ((float)out->contrast / DIVIDE_BY_HUNDRED);

#ifdef DEBUG_OTHERS
	fprintf(stdout, "sat = %f \t bright = %f \t con = %f\n",
			priv->saturation, priv->brightness, priv->contrast);
#endif
}

void calculate_hue_cos_sin_val (PRIVATE *priv)
{
	priv->cos_val = cos (0 * M_PI / 180);
	priv->sin_val = sin (0 * M_PI / 180);

#ifdef DEBUG_OTHERS
	fprintf(stdout, "cos = %f \t sin = %f \n", priv->cos_val, priv->sin_val);
#endif
}

void compute_csc_b_con_matrix (CLRPROP *out, PRIVATE *priv)
{
	int i = 0;

	for (i = 0; i < 3; i++)
		out->csc.B[i] = (CSC_MAX_VALUE - round(priv->contrast * CSC_MAX_VALUE));

#ifdef DEBUG_CSC_B_MATRIX
	for (i = 0; i < 3; i++)
		fprintf(stdout, "Con B[%d] = %d \n", i, out->csc.B[i]);
#endif
}

int clamp (float val)
{
    if (val < CSC_MIN_VALUE)
        val = CSC_MIN_VALUE;

    if (val > CSC_MAX_VALUE)
        val = CSC_MAX_VALUE;

    return (int)val;
}

void compute_csc_a_matrix (CLRPROP *out, PRIVATE *priv)
{
	int i = 0;
	int j = 0;

	float bright_con = (priv->brightness * priv->contrast);

	float bright_con_sat_cos = (priv->brightness * priv->contrast *
								priv->saturation * priv->cos_val);

	float bright_con_sat_sin = (priv->brightness * priv->contrast *
								priv->saturation * priv->sin_val);

	/*
	 * Rounding off the value to nearest integer because
	 * need to pass integer CSC matrix value to IPU-v3
	 * device driver to update matrix
	 */
	out->csc.A[0][0] = round(((0.299 * bright_con) + (0.701 * bright_con_sat_cos) +
						(0.168 * bright_con_sat_sin)) * CSC_MAX_VALUE);

	out->csc.A[0][1] = round((((0.587 * bright_con) - (0.587 * bright_con_sat_cos)) +
						(0.330 * bright_con_sat_sin)) * CSC_MAX_VALUE);

	out->csc.A[0][2] = round((((0.114 * bright_con) - (0.114 * bright_con_sat_cos)) -
						(0.497 * bright_con_sat_sin)) * CSC_MAX_VALUE);

	out->csc.A[1][0] = round((((0.299 * bright_con) - (0.299 * bright_con_sat_cos)) -
						(0.328 * bright_con_sat_sin)) * CSC_MAX_VALUE);

	out->csc.A[1][1] = round(((0.587 * bright_con) + (0.413 * bright_con_sat_cos) +
						(0.035 * bright_con_sat_sin)) * CSC_MAX_VALUE);

	out->csc.A[1][2] = round((((0.114 * bright_con) - (0.114 * bright_con_sat_cos)) +
						(0.292 * bright_con_sat_sin)) * CSC_MAX_VALUE);

	out->csc.A[2][0] = round((((0.299 * bright_con) - (0.3 * bright_con_sat_cos)) +
						(1.25 * bright_con_sat_sin)) * CSC_MAX_VALUE);

	out->csc.A[2][1] = round((((0.587 * bright_con) - (0.588 * bright_con_sat_cos)) -
						(1.05 * bright_con_sat_sin)) * CSC_MAX_VALUE);

	out->csc.A[2][2] = round(((0.114 * bright_con) + (0.886 * bright_con_sat_cos) -
						(0.203 * bright_con_sat_sin)) * CSC_MAX_VALUE);

	/*
	 * Copy CSC 'A' matrix into private drm_imx_csc 'A' matrix,
	 * needed for CSC 'A' matrix calculation w.r.t change in Hue value
	 */
	for (i = 0; i < 3; i++)
		for (j = 0; j < 3; j++)
			priv->cprop.csc.A[i][j] = out->csc.A[i][j];

#ifdef DEBUG_CSC_A_MATRIX
	for (i = 0; i < 3; i++)
	{
		for (j = 0; j < 3; j++)
		{
			fprintf (stdout, "A[%d][%d] = %d \t", i, j, out->csc.A[i][j]);
		}

		printf ("\n");
	}
#endif
}

void compute_csc_a_hue_matrix (CLRPROP *out, PRIVATE *priv)
{
#ifdef DEBUG_CSC_A_MATRIX
	int i = 0;
	int j = 0;
#endif

	/* convert degrees to radians */
	const float cosA = cos(out->hue * M_PI / 180);
	const float sinA = sin(out->hue * M_PI / 180);

/* PRQA: Lint Message 446: Initializer is not modifying any variable */
/*lint -save -e446 */
	/* calculate the rotation matrix, only depends on Hue */
	float matrix[3][3] = {{cosA + (1.0f - cosA) / 3.0f,
						  (1.0f/3.0f) * (1.0f - cosA) - sqrtf(1.0f/3.0f) * sinA,
						  (1.0f/3.0f) * (1.0f - cosA) + sqrtf(1.0f/3.0f) * sinA},
						  {(1.0f/3.0f) * (1.0f - cosA) + sqrtf(1.0f/3.0f) * sinA,
						  cosA + (1.0f/3.0f) * (1.0f - cosA),
						  (1.0f/3.0f) * (1.0f - cosA) - sqrtf(1.0f/3.0f) * sinA},
						  {(1.0f/3.0f) * (1.0f - cosA) - sqrtf(1.0f/3.0f) * sinA,
						  (1.0f/3.0f) * (1.0f - cosA) + sqrtf(1.0f/3.0f) * sinA,
						  cosA + (1.0f/3.0f) * (1.0f - cosA)}};
/*lint -restore*/

	/* Use the rotation matrix to convert to RGB directly */
	out->csc.A[0][0] = clamp (priv->cprop.csc.A[0][0] * matrix[0][0] +
						priv->cprop.csc.A[0][1] * matrix[0][1] +
						priv->cprop.csc.A[0][2] * matrix[0][2]);
	out->csc.A[0][1] = clamp (priv->cprop.csc.A[0][0] * matrix[1][0] +
						priv->cprop.csc.A[0][1] * matrix[1][1] +
						priv->cprop.csc.A[0][2] * matrix[1][2]);
	out->csc.A[0][2] = clamp (priv->cprop.csc.A[0][0] * matrix[2][0] +
						priv->cprop.csc.A[0][1] * matrix[2][1] +
						priv->cprop.csc.A[0][2] * matrix[2][2]);

	out->csc.A[1][0] = clamp (priv->cprop.csc.A[1][0] * matrix[0][0] +
						priv->cprop.csc.A[1][1] * matrix[0][1] +
						priv->cprop.csc.A[1][2] * matrix[0][2]);
	out->csc.A[1][1] = clamp (priv->cprop.csc.A[1][0] * matrix[1][0] +
						priv->cprop.csc.A[1][1] * matrix[1][1] +
						priv->cprop.csc.A[1][2] * matrix[1][2]);
	out->csc.A[1][2] = clamp (priv->cprop.csc.A[1][0] * matrix[2][0] +
						priv->cprop.csc.A[1][1] * matrix[2][1] +
						priv->cprop.csc.A[1][2] * matrix[2][2]);

	out->csc.A[2][0] = clamp (priv->cprop.csc.A[2][0] * matrix[0][0] +
						priv->cprop.csc.A[2][1] * matrix[0][1] +
						priv->cprop.csc.A[2][2] * matrix[0][2]);
	out->csc.A[2][1] = clamp (priv->cprop.csc.A[2][0] * matrix[1][0] +
						priv->cprop.csc.A[2][1] * matrix[1][1] +
						priv->cprop.csc.A[2][2] * matrix[1][2]);

	out->csc.A[2][2] = clamp (priv->cprop.csc.A[2][0] * matrix[2][0] +
						priv->cprop.csc.A[2][1] * matrix[2][1] +
						priv->cprop.csc.A[2][2] * matrix[2][2]);

#ifdef DEBUG_HUE_MATRIX
	for (i = 0; i < 3; i++)
	{
		for (j = 0; j < 3; j++)
		{
			fprintf (stdout, "matrix[%d][%d] = %f \t", i, j, matrix[i][j]);
		}

		printf ("\n");
	}
#endif
#ifdef DEBUG_CSC_A_MATRIX
	for (i = 0; i < 3; i++)
	{
		for (j = 0; j < 3; j++)
		{
			fprintf (stdout, "A[%d][%d] = %d \t", i, j, out->csc.A[i][j]);
		}

		printf ("\n");
	}
#endif
}

void calculate_hue(CLRPROP *out, PRIVATE *priv, int arr_index, int rem, int offset)
{
	float val = 0;
	float temp = 0;
	int b_zero_val = priv->cprop.csc.B[0];
	int b_one_val = priv->cprop.csc.B[1];

	val = ((float)rem / HUE_RANGE_VAL);
	temp = val;

	if (out->csc.B[arr_index] >= 0)
		val = (offset * temp);

	if(0 == arr_index)
	{
		out->csc.B[arr_index] = round(out->csc.B[arr_index] - val);
		if (out->csc.B[arr_index] < 0)
		{
			rem = round(b_zero_val / temp);
			rem = round((offset - rem) * temp);
			out->csc.B[arr_index] = (SIGNED_FOURTEEN_BIT_VAL - rem);
		}
	}
	else
	{
		out->csc.B[arr_index] = round(out->csc.B[arr_index] + val);
		if (out->csc.B[arr_index] >= SIGNED_FOURTEEN_BIT_VAL)
		{
			out->csc.B[arr_index] = 0;
			rem = round((SIGNED_FOURTEEN_BIT_VAL - b_one_val) / temp);
			rem = round((offset - rem) * temp);
			out->csc.B[arr_index] += rem;
		}
	}
}

void compute_csc_b_hue_matrix (CLRPROP *out, PRIVATE *priv)
{
	int i = 0;
	int rem = 0;

	if (out->csc.B[i + 1] < out->csc.B[i])
		rem = out->csc.B[i] - out->csc.B[i + 1];
	else if (out->csc.B[i + 1] > out->csc.B[i])
		rem = (SIGNED_FOURTEEN_BIT_VAL - out->csc.B[i + 1]) + out->csc.B[i];
	else
		rem = 0;

	if (rem > 0)
	{
		if ((out->hue_off >= HUE_OFF_MIN_VAL) && (out->hue_off <= -1))
		{
			if ((out->hue_off >= -60) && (out->hue_off <= -1))
				calculate_hue(out, priv, (i + 2), rem, (-out->hue_off));

			if ((out->hue_off >= -120) && (out->hue_off <= -61))
			{
				calculate_hue(out, priv, (i + 2), rem, HUE_RANGE_VAL);
				calculate_hue(out, priv, i, rem, (-(out->hue_off - (-HUE_RANGE_VAL))));
			}

			if ((out->hue_off >= HUE_OFF_MIN_VAL) && (out->hue_off <= -121))
			{
				calculate_hue(out, priv, (i + 2), rem, HUE_RANGE_VAL);
				calculate_hue(out, priv, i, rem, HUE_RANGE_VAL);
				calculate_hue(out, priv, (i + 1), rem, (-(out->hue_off - (-HUE_RANGE_VAL * 2))));
			}
		}

		if ((out->hue_off <= HUE_OFF_MAX_VAL) && (out->hue_off >= 1))
		{
			if ((out->hue_off <= 60) && (out->hue_off >= 1))
				calculate_hue(out, priv, (i + 1), rem, out->hue_off);

			if ((out->hue_off <= 120) && (out->hue_off >= 61))
			{
				calculate_hue(out, priv, (i + 1), rem, HUE_RANGE_VAL);
				calculate_hue(out, priv, i, rem, (out->hue_off - HUE_RANGE_VAL));
			}

			if ((out->hue_off <= HUE_OFF_MAX_VAL) && (out->hue_off >= 121))
			{
				calculate_hue(out, priv, (i + 1), rem, HUE_RANGE_VAL);
				calculate_hue(out, priv, i, rem, HUE_RANGE_VAL);
				calculate_hue(out, priv, (i + 2), rem, (out->hue_off - (HUE_RANGE_VAL * 2)));
			}
		}
	}

#ifdef DEBUG_CSC_B_MATRIX
	for (i = 0; i < 3; i++)
		fprintf(stdout, "Hue B[%d] = %d \n", i, out->csc.B[i]);
#endif
}

void calculate_saturation(CLRPROP *out, PRIVATE *priv, float val)
{
	int i = 0;
	int rem = 0;
	int contrast_val = 0;
	float temp = val;

	contrast_val = (CSC_MAX_VALUE - round(priv->contrast * CSC_MAX_VALUE));

	for(i = 1; i < 3; i++)
	{
		if (out->csc.B[i] >= 0)
			val = (out->saturation_off * temp);

		out->csc.B[i] = round((out->csc.B[i] + contrast_val) -
							(val + out->csc.B[i]));
		if (out->csc.B[i] < 0)
		{
			rem = round(out->csc.B[i] / temp);
			rem = -(rem);
			out->csc.B[i] = round(SIGNED_FOURTEEN_BIT_VAL -
										(rem * temp));
		}

		/* Restore values for next iterations */
		val = temp;
	}
}

void calculate_saturation_neg(CLRPROP *out, float val)
{
	int i = 0;
	int rem = 0;
	float temp = val;

	for(i = 1; i < 3; i++)
	{
		if (out->csc.B[i] >= 0)
			val = (out->saturation_off * temp);

		out->csc.B[i] = round(out->csc.B[i] - val);
		if (out->csc.B[i] < 0)
		{
			rem = round(out->csc.B[i] / temp);
			rem = -(rem);
			out->csc.B[i] = round(SIGNED_FOURTEEN_BIT_VAL -
										(rem * temp));
		}

		/* Restore values for next iterations */
		val = temp;
	}
}

void calculate_val_saturation(CLRPROP *out)
{
	float val = 0;

	val = (((float)out->brightness_off / DIVIDE_BY_HUNDRED) *
			CSC_B_MATRIX_MAX_VALUE);
	val = (CSC_B_MATRIX_MAX_VALUE - (-val));
	val = ((val / DIVIDE_BY_HUNDRED) * out->saturation_off);
	val = (val / out->saturation_off);

	calculate_saturation_neg(out, val);
}

void compute_csc_b_sat_matrix (CLRPROP *out, PRIVATE *priv)
{
	int i = 0;
	float val = 0;

	val = ((float)CSC_B_MATRIX_MAX_VALUE / DIVIDE_BY_HUNDRED);

	if (priv->csc_b_matrix_update && (out->brightness_off >= 0))
		calculate_saturation(out, priv, val);

	else if (priv->csc_b_matrix_update && (out->brightness_off < 0))
		calculate_val_saturation(out);
	else
	{
		for(i = 1; i < 3; i++)
			out->csc.B[i] = round(SIGNED_FOURTEEN_BIT_VAL -
								(out->saturation_off * val));
	}

	/*
	 * Copy CSC 'B' matrix, used for computing matrix w.r.t
	 * change in hue offset
	 */
	for (i = 0; i < 3; i++)
		priv->cprop.csc.B[i] = out->csc.B[i];

#ifdef DEBUG_CSC_B_MATRIX
	for (i = 0; i < 3; i++)
		fprintf(stdout, "Sat B[%d] = %d \n", i, out->csc.B[i]);
#endif
}

void calculate_brightness(CLRPROP *out, float val)
{
	int i = 0;
	int rem = 0;
	float temp = val;

	for (i = 0; i < 3; i++)
	{
		if (out->csc.B[i] >= 0)
			val = ((-out->brightness_off) * temp);

		out->csc.B[i] = round(out->csc.B[i] - val);
		if (out->csc.B[i] < 0)
		{
			rem = round(out->csc.B[i] / temp);
			rem = -(rem);
			out->csc.B[i] = round(SIGNED_FOURTEEN_BIT_VAL - (rem * temp));
		}

		/* Restore values for next iterations */
		val = temp;
	}

}

void compute_csc_b_bright_matrix (CLRPROP *out)
{
	int i = 0;
	float val = 0;

	val = ((float)CSC_B_MATRIX_MAX_VALUE / DIVIDE_BY_HUNDRED);

	if ((out->brightness_off >= BRIGHTNESS_OFF_MIN_VAL) &&
			(out->brightness_off <= -1))
		calculate_brightness(out, val);

	if ((out->brightness_off <= BRIGHTNESS_OFF_MAX_VAL) &&
			(out->brightness_off >= 1))
	{
		for(i = 0; i < 3; i++)
			out->csc.B[i] = round(out->brightness_off * val);
	}

#ifdef DEBUG_CSC_B_MATRIX
	for (i = 0; i < 3; i++)
		fprintf(stdout, "Brightness B[%d] = %d \n", i, out->csc.B[i]);
#endif
}

void compute_csc_matrix(CLRPROP *out, PRIVATE *priv)
{
	/*
	 * Now will not consider HUE and it's offset value change
	 * for CSC matrix computation i.e. Hue will always be '0' degree.
	 * First will divide color properties by 100 and compute matrix
	 * accordingly
	 */
	div_color_prop (out, priv);

	/* Need to calculate cosine and sine value for 0 degree Hue */
	calculate_hue_cos_sin_val (priv);

	/*
	 * CSC A matrix computation w.r.t change in
	 * saturation, brightness and contrast
	 */
	compute_csc_a_matrix (out, priv);

	/* Hue matrix */
	compute_csc_a_hue_matrix (out, priv);

	priv->csc_a_matrix_update = true;

	if (DEFAULT_CONTRAST_VAL != out->contrast)
		priv->contrast_update = true;

	/* if change in contrast value then need to compute CSC 'B' matrix */
	if (priv->contrast_update)
	{
		compute_csc_b_con_matrix (out, priv);

		if (!priv->csc_b_matrix_update)
			priv->csc_b_matrix_update = true;
	}

	if (DEFAULT_BRIGHTNESS_OFF_VAL != out->brightness_off)
	{
		compute_csc_b_bright_matrix (out);

		if (!priv->csc_b_matrix_update)
			priv->csc_b_matrix_update = true;
	}

	/*
	 * No need to compute CSC 'B' matrix w.r.t change in
	 * saturation offset value when brightness offset value
	 * is minimum (-100)
	 */
	if ((DEFAULT_SATURATION_OFF_VAL != out->saturation_off) &&
			(BRIGHTNESS_OFF_MIN_VAL != out->brightness_off))
	{
		compute_csc_b_sat_matrix (out, priv);

		if (!priv->csc_b_matrix_update)
			priv->csc_b_matrix_update = true;
	}

	/*
	 * No need to compute CSC 'B' matrix w.r.t change in
	 * hue offset value when brightness offset value
	 * is minimum (-100)
	 */
	if (priv->hue_off_update &&
		(BRIGHTNESS_OFF_MIN_VAL != out->brightness_off))
	{
		compute_csc_b_hue_matrix (out, priv);

		if (!priv->csc_b_matrix_update)
			priv->csc_b_matrix_update = true;
	}
}

int retrieve_crtc_id (int32_t drm_fd, PRIVATE *priv)
{
	int i = 0;
	int j = 0;
	int res = SUCCESS;
	uint32_t cnt_planes = 0;
	drmModeCrtcPtr *crtcs = NULL;
	drmModeRes *resources;
	drmModePlaneRes *plane_resources;
	drmModePlane *plane;

	resources = drmModeGetResources(drm_fd);
	if (!resources)
	{
		fprintf(stdout, "drmModeGetResources failed\n");
		res = ERROR;
		goto err;
	}
	else
	{
		crtcs = (drmModeCrtcPtr *) calloc(resources->count_crtcs,
					sizeof(drmModeCrtcPtr));
		if (crtcs != NULL)
		{
			priv->crtc_count = resources->count_crtcs;

			for (i = 0; i < resources->count_crtcs && i < MAX_CRTC_ID; i++)
				crtcs[i] = drmModeGetCrtc(drm_fd, resources->crtcs[i]);

			plane_resources = drmModeGetPlaneResources(drm_fd);
			if (!plane_resources)
			{
				fprintf(stdout, "drmModeGetPlaneResources failed\n");
				res = ERROR;
				goto free_crtcs;
			}

			for (cnt_planes = 0; cnt_planes < plane_resources->count_planes;
					cnt_planes++)
			{
				plane = drmModeGetPlane(drm_fd,
						plane_resources->planes[cnt_planes]);
				if (plane != NULL)
				{
					for (j = 0; j < resources->count_crtcs; j++)
					{
						if (plane->possible_crtcs & (1 << j))
							if (crtcs[j] != NULL)
								priv->crtc_id[j] = crtcs[j]->crtc_id;
					}

					drmModeFreePlane(plane);
				}
				else
				{
					fprintf(stdout, "Failed to retrieve plane \n");
					res = ERROR;
					goto free_plane_resource;
				}
			}

			/*
			 * Free the resources, plane, crtc's and
			 * dynamic memory allocated
			 */
			goto free_plane_resource;
		}
		else
		{
			fprintf(stdout, "Failed to allocate memory, to retrieve crtc information \n");
			res = ERROR;
			goto err;
		}
	}

free_plane_resource:
	drmModeFreePlaneResources(plane_resources);

free_crtcs:
	for (i = 0; i < resources->count_crtcs; i++)
	{
		if (crtcs[i] != NULL)
			drmModeFreeCrtc(crtcs[i]);
	}
	free(crtcs);
	drmModeFreeResources(resources);

err:
	return res;
}

int retrieve_config_crtc_type (int32_t drm_fd, CLRPROP *out, PRIVATE *priv)
{
	int res = SUCCESS;
	uint32_t i = 0;
	uint32_t count = 0;

	/* CSC enable will always be true */
	out->csc.enable = true;

	/* By default CSC type will be "DRM_IMX_CSC_RGB2RGB"
	 * else library will use clr_space_type value passed
	 * from the application.
	 */
	switch (priv->type)
	{
		case RGB2RGB:
			out->csc.type = DRM_IMX_CSC_RGB2RGB;
			break;

		case RGB2YUV:
			out->csc.type = DRM_IMX_CSC_RGB2YUV;
			break;

		case YUV2RGB:
			out->csc.type = DRM_IMX_CSC_YUV2RGB;
			break;

		case YUV2YUV:
			out->csc.type = DRM_IMX_CSC_YUV2YUV;
			break;

		default:
			out->csc.type = DRM_IMX_CSC_RGB2RGB;
			break;
	}

	/* By default crtc id will be retrieved in libipu_dp_csc.so library
	 * if crct_id passed with value "0xFF" else library will use the
	 * crtc id passed from the application.
	 */
	if (priv->crtc_update)
		/* Retrieve the crtc id and update drm_imx_csc crtc id member */
		res = retrieve_crtc_id(drm_fd, priv);
	else
	{
		for (i = 0; i < priv->crtc_count; i++)
			if (priv->cprop.csc.crtc_id != priv->crtc_id[i])
				count++;

		/*
		 * Count reached its maximum value. Crtc id passed from
		 * the application is not matching with the retrieved dual
		 * plane crtc id. Return invalid crtc error
		 */
		if (count == priv->crtc_count)
			res = ERROR;
		else
			out->csc.crtc_id = priv->cprop.csc.crtc_id;
	}

	return res;
}

int update_csc_matrix (int32_t drm_fd, CLRPROP *out, PRIVATE *priv)
{
	int res = ERROR;
	uint32_t i = 0;
	uint32_t j = 0;

	if (priv->crtc_update)
	{
		for (i = 0; i < priv->crtc_count; i++)
		{
			if (INVALID_VALUE != priv->crtc_id[i])
			{
				out->csc.crtc_id = priv->crtc_id[i];

				if (priv->csc_b_matrix_update || priv->csc_a_matrix_update)
					res = drmIoctl(drm_fd, DRM_IOCTL_IMX_SET_CSC, &out->csc);

				if (res != ERROR)
					out->supported_crtc_id[j++] = priv->crtc_id[i];
			}
		}
	}
	else
	{
		if (priv->csc_b_matrix_update || priv->csc_a_matrix_update)
			res = drmIoctl(drm_fd, DRM_IOCTL_IMX_SET_CSC, &out->csc);
	}

	return res;
}

void csc_update_crtc_id (uint32_t crtc_id)
{
	g_priv.cprop.csc.crtc_id = crtc_id;
}

void csc_update_color_space_conversion_type (CLRSPACETYPE type)
{
	g_priv.type = type;
}

void copy_clr_prop_val(CLRPROP *out, PRIVATE *priv)
{
	priv->cprop.hue = out->hue;
	priv->cprop.saturation = out->saturation;
	priv->cprop.brightness = out->brightness;
	priv->cprop.contrast = out->contrast;

	priv->cprop.hue_off = out->hue_off;
	priv->cprop.saturation_off = out->saturation_off;
	priv->cprop.brightness_off = out->brightness_off;

#ifdef DEBUG_CLR_PROPERTY
	fprintf (stdout, "Hue = %d \n"
				"Saturation = %d \n"
				"Brightness = %d \n"
				"Contrast = %d \n"
				"Hue offset = %d \n"
				"Saturation offset = %d \n"
				"Brightness offset = %d\n", priv->cprop.hue,
				priv->cprop.saturation, priv->cprop.brightness,
				priv->cprop.contrast, priv->cprop.hue_off,
				priv->cprop.saturation_off, priv->cprop.brightness_off);
#endif
}

int csc_dp_compute_update_matrix (int32_t drm_fd, bool csc_update, CLRPROP *out)
{
#ifdef DEBUG_CSC_A_B_MATRIX
	int i = 0;
	int j = 0;
#endif
	int res = SUCCESS;

	if (NULL == out)
	{
		fprintf(stdout, "Incorrect structure passed !!!\n");
		return ERROR;
	}

	if (ERROR == drm_fd)
	{
		fprintf(stdout, "Wrong drm handle passed !!!\n");
		return ERROR;
	}

	/* Initialization */
	initialize_clrprop_priv (out, &g_priv);

	correct_color_property_val (out, &g_priv);

	copy_clr_prop_val(out, &g_priv);

	compute_csc_matrix (out, &g_priv);

	if (csc_update)
	{
		res = retrieve_config_crtc_type(drm_fd, out, &g_priv);

		if (SUCCESS == res)
		{
			res = update_csc_matrix (drm_fd, out, &g_priv);
			if (res < SUCCESS)
				fprintf (stdout, "Error in CSC matrix update !!! \n");
			else
				fprintf (stdout, "Successfully updated CSC matrix !!! \n");
		}
		else
		{
			if (!g_priv.crtc_update)
				fprintf(stdout, "Invalid crtc id passed \n");
		}
	}

	return res;
}

void set_default_clr_property_val(CLRPROP *out)
{
	out->hue = DEFAULT_HUE_VAL;
	out->saturation = DEFAULT_SATURATION_VAL;
	out->brightness = DEFAULT_BRIGHTNESS_VAL;
	out->contrast = DEFAULT_CONTRAST_VAL;

	/* offset */
	out->hue_off = DEFAULT_HUE_OFF_VAL;
	out->saturation_off = DEFAULT_SATURATION_OFF_VAL;
	out->brightness_off = DEFAULT_BRIGHTNESS_OFF_VAL;

	/*
	 * Make a copy of the default color property values
	 * passed to the application. These values
	 * will be used when corresponding color property
	 * value passed from application is of "INVALID_VALUE"
	 */
	copy_clr_prop_val(out, &g_priv);
}

void csc_set_default_value (CLRPROP *out)
{
	int i = 0;
	int a[3][3] = {
					{256, 0, 0},
					{0, 256, 0},
					{0, 0, 256}
				  };

	/*
	 * Set default color property values and
	 * share the values with application for
	 * future tuning
	 */
	set_default_clr_property_val(out);

	memcpy(out->csc.A, a, sizeof(out->csc.A));

	/*
	 * CSC 'E' matrix will not change forever
	 * so set 'E' matrix with default value '2'.
	 * By default CSC 'B' matrix will be configured with
	 * value '0', will change based on hue, saturation and
	 * brightness offset values
	 */
	for(i = 0; i < 3; i++)
	{
		out->csc.E[i] = CSC_EXPONENT_VAL;
		out->csc.B[i] = CSC_MIN_VALUE;
	}
}
/*lint -restore*/
